/* 
 * print-limits.c 
 *
 * Виводить діючі обмеження на використання ресурсів.
 * Ілюструє порядок застосування функції getrlimit().
 *
 */

#include <errno.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/resource.h>
#include <unistd.h>

static void showvalue(rlim_t lim);
static char showlimit(int resource, const char *name);


static void showvalue(rlim_t lim)
{
        /* У деяких системах макроси RLIM_SAVED_CUR, RLIM_SAVED_MAX
           повертають RLIM_INFINITY, тому оператор switch використовувати
           не можна, RLIM_INFINITY треба перевіряти першим. */
        if (lim == RLIM_INFINITY)
                printf("RLIM_INFINITY");
        else if (lim == RLIM_SAVED_CUR)
                printf("RLIM_SAVED_CUR");
        else if (lim == RLIM_SAVED_MAX)
                printf("RLIM_SAVED_MAX");
        else
                printf("%llu", (unsigned long long) lim);
}

static char showlimit(int resource, const char *name)
{
        struct rlimit r;

        if (getrlimit(resource, &r) != 0) {
                fprintf(stderr, "Error getting limit for %s: %s\n",
                                                name, strerror(errno));
                return 0;
        }
        printf("%s: ", name);
        printf("rlim_cur = ");
        showvalue(r.rlim_cur);
        printf("; rlim_max = ");
        showvalue(r.rlim_max);
        printf("\n");
        return 1;
}

int main()
{
        /* Розмір rlim_t стандарт SUS3 не визначає. */
        if (sizeof(rlim_t) > sizeof(long long))
                printf("Warning: rlim_t > long long;"
                                " results may be wrong\n");

        /* Виводить діючі обмеження на використання ресурсів. */
        if (!showlimit(RLIMIT_CORE, "RLIMIT_CORE")
                || !showlimit(RLIMIT_CPU, "RLIMIT_CPU")
                || !showlimit(RLIMIT_DATA, "RLIMIT_DATA")
                || !showlimit(RLIMIT_FSIZE, "RLIMIT_FSIZE")
                || !showlimit(RLIMIT_NOFILE, "RLIMIT_NOFILE")
                || !showlimit(RLIMIT_STACK, "RLIMIT_STACK")
                || !showlimit(RLIMIT_AS, "RLIMIT_AS")) {
                exit(EXIT_FAILURE);
        }

        exit(EXIT_SUCCESS);
}
